'use strict';

angular.module('genisis').controller('editRequestCtrl', ['$scope', 'request',
  '$rootScope', 'source', 'type', 'studyApproval', 'api', 'user',
  '$location', '$routeParams', '$window', 'focus',

  function ($scope, $requests, $rootScope, $sources, $types,
    $studyApprovals, $api, $user, $location, $routeParams, $window, focus) {

    $scope.requests = $requests;
    $scope.request = {};
    $scope.studies = {};
    $scope.studyApprovals = $studyApprovals;
    $scope.sources = $sources;
    $scope.types = $types;
    $scope.editable = false;
    $scope.submitted = false;
    $scope.commentable = false;
    $scope.canCopyTable = false;
    $scope.sent = false;
    $scope.accepted = false;
    $scope.delivered = false;
    $scope.logs = [];
    $scope.loadingRequest = true;
    $scope.loadingLogs = true;
    $scope.copyTableEnabled = false;
    $scope.copyingTable = false;
    $scope.copyTableData = {};
    $scope.successCopiedTable = false;
    $scope.failedCopiedTable = false;
    $scope.copyTableMessage = '';

    $scope.visibleComments = [{
        comment: 'Please check that you have the correct study selected,' +
          'it seems like this will not work with the current ' +
          'study you selected.',
        user: 'DataManager1',
        date: '12:50:30 03/07/2017'
      },
      {
        comment: 'Sure, will do!',
        user: 'Researcher1',
        date: '01:30:05 03/07/2017'
      },
    ];

    $studyApprovals.load().then(function (loadedApprovals) {
      loadedApprovals.unshift({
        title: 'None',
        id: null
      });
      $scope.studies = loadedApprovals;
    });

    /**
     * Loads data for a specific request
     * @param int requestID the ID of the request to load
     * @return boolean
     **/
    $scope.loadRequest = function (requestID) {

      //load the request matching the given id
      $requests.load().then(function ( /*request*/ ) {
        $scope.request = $requests.find(requestID);
        switch ($scope.request.statusDescription) {
        case 'Draft':
          $scope.editable = true;
          $scope.commentable = false;
          break;
        case 'Returned':
          $scope.editable = true;
          $scope.commentable = true;
          break;
        case null:
          $scope.editable = false;
          break;
        case 'Submitted':
          if ($user.destinationMgr() || $user.admin()) {
            $scope.submitted = true;
          }
          $scope.commentable = true;
          break;
        case 'Sent':
          if ($user.sourceMgr() || $user.admin()) {
            $scope.sent = true;
          }
          $scope.commentable = true;
          break;
          // DSM
        case 'RequestAccepted':
          if ($user.sourceMgr() || $user.admin()) {
            $scope.accepted = true;
          }
          break;
        case 'ResultsDelivered':
          $scope.delivered = true;
          $scope.commentable = true;
          break;
        }

        // if ($scope.request.reason !== undefined &&
        //   $scope.request.reason !== null) {
        //   $rootScope.warnings.push($scope.request.reason);
        // }

        return true;
      });
    };

    $scope.resetForm = function () {
      $scope.request.comments = null;
    };

    $scope.loadRequestByID = function (requestID) {
      $requests.findByID(requestID).then(function (request) {
        $scope.request = request;

        switch ($scope.request.statusDescription) {
        case 'Draft':
          $scope.editable = true;
          $scope.commentable = false;
          break;
        case 'Returned':
          if ($scope.canEdit($user, request)) {
            $scope.editable = true;
          } else {
            $scope.editable = false;
          }
          $scope.commentable = true;
          break;
        case null:
          $scope.editable = false;
          break;
        case 'Submitted':
          if ($user.destinationMgr() || $user.admin()) {
            $scope.submitted = true;
            $scope.commentable = true;
          }
          break;
        case 'Sent':
          if ($user.sourceMgr() || $user.admin()) {
            $scope.sent = true;
            $scope.commentable = true;
          }
          break;
        case 'RequestAccepted':
          if ($user.sourceMgr() || $user.admin()) {
            $scope.accepted = true;
            $scope.commentable = true;
          }
          break;
        case 'ResultsDelivered':
          if ($user.destinationMgr() || $user.admin()) {
            $scope.delivered = true;
            $scope.setTableCopy(request);
            $scope.commentable = true;
          }
          break;
        }

        if ($location.search().created === 'true') {
          $window.alert('Request created successfully.');
        }

        // if ($scope.request.reason !== undefined &&
        //   $scope.request.reason !== null) {
        //   $rootScope.warnings.push($scope.request.reason);
        // }

        $scope.loadingRequest = false;

        $scope.setCommentability(request);

        return true;
      });
    };

    $scope.loadLogs = function (requestID) {

      //load the logs for this request
      $requests.logs(requestID).then(function (logData) {
        $scope.logs = $requests.parseLogs(logData);
        if ($scope.logs) {
          $scope.loadingLogs = false;
        }

      });
    };

    //load the request matching the given id
    $scope.studies = $studyApprovals.load();

    /**
     * Save updates made to a request
     * @param Object request
     * @return boolean
     **/
    $scope.saveRequest = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if ($requests.exceededMaxCharacters(req.description)) {
        // $rootScope.errors.push(
        //   'Description exceeds 3000 characters'
        // );

        $window.alert('Description exceeds 3000 characters');

        return false;
      }

      if (!$requests.valid(req)) {
        $window.alert('Please fill and select required fields');
        return false;
      }

      return $api.update('requests/' + req.id, {
        data: req
      }).then(function (request) {

        if (request.success && request.response) {
          // $rootScope.messages.push('Your changes have been saved.');
          $window.alert('Your changes have been saved.');
          return true;
        } else {
          // $rootScope.errors.push('Save request API call failed.');
          $api.errorAlert();
        }

        return false;
      });
    };

    /**
     * Submitting a request to the data manager
     * @param Object request data
     * @return boolean
     **/
    $scope.submitRequest = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if ($requests.exceededMaxCharacters(req.description)) {
        // $rootScope.errors.push(
        //   'Description exceeds 3000 characters'
        // );
        $window.alert('Description exceeds 3000 characters');

        return false;
      }

      if (!$requests.valid(req)) {
        return false;
      }

      return $api.update('requests/' + req.id + '/submit', {
        data: req
      }).then(function (request) {

        if (request.success && request.response) {

          //this is no longer editable
          $scope.editable = false;
          req.statusDescription = 'Submitted';

          //set a message that the request was submitted
          // $rootScope.messages.push(
          //   'Your request has been submitted to your data manager.'
          // );
          $window.alert(
            'Your request has been submitted to your data manager.'
          );

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else {
          // $rootScope.errors.push('Submit request API call failed.');
          $api.errorAlert();
        }

        return false;
      });
    };

    /**
     * Approve a request
     * @param Object request
     * @return boolean
     **/
    $scope.approveRequest = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      var data = {
        createdBy: $user.id()
      };

      if ($requests.exceededMaxCharacters(req.comments)) {
        // $rootScope.errors.push(
        //   'Comment exceeds 3000 characters'
        // );

        $window.alert('Comment exceeds 3000 characters');

        return false;
      }

      if (req.comments && req.comments.length > 0) {
        data.comments = req.comments;
      }

      return $api.update('requests/' + req.id + '/sent', {
        data: data
      }).then(function (request) {

        if (request.success && request.response) {
          // $rootScope.messages.push('The request has been approved.');
          $window.alert('The request has been approved.');
          $scope.editable = false;
          $scope.submitted = false;
          if ($user.sourceMgr() || $user.admin()) {
            $scope.sent = true;
          }
          req.statusDescription = 'Sent';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else {
          // $rootScope.errors.push('Approve request API call failed.');
          $api.errorAlert();
        }

        return false;
      });
    };

    /**
     * Save updates made to a request
     * @param string message why this request is being rejected
     * @param Object request the request that's being returned
     * @return boolean
     **/
    $scope.returnRequest = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.message = [];

      if ($requests.exceededMaxCharacters(message)) {
        // $rootScope.errors.push(
        //   'Comment exceeds 3000 characters'
        // );
        $window.alert('Comment exceeds 3000 characters');

        return false;
      }

      if (!message) {
        // $rootScope.errors.push(
        //   'Enter a reason why you\'re denying this request in ' +
        //   'the comment field.'
        // );

        $window.alert('Enter a reason why you\'re returning this' +
          ' request in the comment field.');

        return false;
      }

      return $api.update('requests/' + req.id + '/return', {
        data: {
          comments: message,
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          // $rootScope.messages.push(
          //   'This request has been returned to the owner.'
          // );
          $window.alert(
            'This request has been returned to the owner.');
          $scope.submitted = false;
          req.statusDescription = 'Returned';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else {
          // $rootScope.errors.push('Reject request API call failed.');
          $api.errorAlert();
        }

        return false;
      });
    };

    /**
     * Allow data managers and admins to deny submitted requests
     * @param string message why this request is being denied
     * @param Object request the request that's being denied
     * @return boolean
     **/
    $scope.denyRequest = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if ($requests.exceededMaxCharacters(message)) {
        // $rootScope.errors.push(
        //   'Comment exceeds 3000 characters'
        // );

        $window.alert('Comment exceeds 3000 characters');

        return false;
      }

      if (!message) {
        // $rootScope.errors.push(
        //   'Enter a reason why you\'re denying this request in ' +
        //   'the comment field.'
        // );

        $window.alert('Enter a reason why you\'re denying' +
          ' this request in the comment field.');

        return false;
      }

      return $api.update('requests/' + req.id + '/deny', {
        data: {
          comments: message,
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          // $rootScope.messages.push('This request has been denied.');
          $window.alert('This request has been denied.');
          $scope.editable = false;
          $scope.submitted = false;
          req.statusDescription = 'Denied';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else {
          // $rootScope.errors.push('Deny request API call failed.');
          $api.errorAlert();
        }

        return false;
      });
    };

    /**
     * Allow data managers to accept data from the data supplier
     * @param Object request the request that's data is being accepted
     * @return boolean
     **/
    $scope.acceptData = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if ($requests.exceededMaxCharacters(req.comments)) {
        // $rootScope.errors.push(
        //   'Comment exceeds 3000 characters'
        // );
        $window.alert('Comment exceeds 3000 characters');

        return false;
      }

      var data = {
        createdBy: $user.id()
      };

      if (req.comments && req.comments.length > 0) {
        data.comments = req.comments;
      }

      return $api.update('requests/' + req.id + '/acceptdata', {
        data: data
      }).then(function (request) {

        if (request.success && request.response) {
          // $rootScope.messages.push(
          //   'This request\'s data has been accepted.');
          $window.alert('This request\'s data has been accepted.');
          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = true;
          req.statusDescription = 'RequestAccepted';

          $scope.setTableCopy(request.response);
          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else {
          // $rootScope.errors.push(
          //   'Accept data request API call failed.');
          $api.errorAlert();
        }

        return false;
      });
    };

    /**
     * Allow data managers to reject data from the data supplier
     * @param Object request the request that's data is being accepted
     * @return boolean
     **/
    $scope.denyData = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if ($requests.exceededMaxCharacters(message)) {
        // $rootScope.errors.push(
        //   'Comment exceeds 3000 characters'
        // );
        $window.alert('Comment exceeds 3000 characters');

        return false;
      }

      if (!message) {
        // $rootScope.errors.push(
        //   'Enter a reason why you\'re denying this request in ' +
        //   'the comment field.'
        // );
        $window.alert(
          'Enter a reason why you\'re denying this request in ' +
          'the comment field.'
        );

        return false;
      }

      return $api.update('requests/' + req.id + '/denydata', {
        data: {
          comments: message,
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          // $rootScope.messages.push(
          //   'This request has been denied because the ' +
          //   'request cannot be fulfilled.');
          $window.alert(
            'This request has been denied because the ' +
            'request cannot be fulfilled.');

          $scope.editable = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = false;

          req.statusDescription = 'RequestNotAccepted';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else {
          // $rootScope.errors.push('Deny data request API call failed.');
          $api.errorAlert();
        }

        return false;
      });
    };

    $scope.deliverResults = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if ($requests.exceededMaxCharacters(message)) {
        // $rootScope.errors.push(
        //   'Comment exceeds 3000 characters'
        // );

        $window.alert('Comment exceeds 3000 characters');

        return false;
      }

      if (!message) {
        // $rootScope.errors.push(
        //   'Enter required information before delivering these results ' +
        //   'in the comment field.'
        // );

        $window.alert(
          'Enter required information before delivering these results ' +
          'in the comment field.'
        );

        return false;
      }

      return $api.update('requests/' + req.id + '/resultsdelivered', {
        data: {
          comments: message,
          createdBy: $user.id()
        }
      }).then(function (request) {
        if (request.success && request.response) {
          // $rootScope.messages.push(
          //   'Results are delivered.'
          // );
          $window.alert('Results are delivered.');
          if (!$user.sourceMgr()) {
            $scope.delivered = true;
          }
          $scope.accepted = false;
          $scope.sent = false;
          $scope.disableCopyTable();

          req.statusDescription = 'ResultsDelivered';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else {
          // $rootScope.errors.push(
          //   'Deliver results request API call failed.');
          $api.errorAlert();
        }

        return false;
      });
    };

    /**
     * Allow researcher to accept data from the data supplier
     * @param Object request the request that's data is being confirmed for
     * @return boolean
     **/
    $scope.acceptResults = function (req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if ($requests.exceededMaxCharacters(req.comments)) {
        // $rootScope.errors.push(
        //   'Comment exceeds 3000 characters'
        // );
        $window.alert('Comment exceeds 3000 characters');

        return false;
      }

      var data = {
        createdBy: $user.id()
      };

      if (req.comments && req.comments.length > 0) {
        data.comments = req.comments;
      }

      return $api.update('requests/' + req.id + '/confirmdata', {
        data: data
      }).then(function (request) {

        if (request.success && request.response) {
          // $rootScope.messages.push(
          //   'This request\'s data has been accepted.'
          // );
          $window.alert(
            'This request\'s data has been accepted.'
          );
          $scope.editable = false;
          $scope.delivered = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = false;
          $scope.disableCopyTable();
          req.statusDescription = 'ResultsAccepted';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;

        } else {
          // $rootScope.errors.push(
          //   'Accept results request API call failed.');
          $api.errorAlert();
        }

        return false;
      });
    };

    /**
     * Allow researcher to reject the data from the data supplier
     * @param Object request the request that's data is being rejected for
     * @return boolean
     **/
    $scope.denyResults = function (message, req) {

      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if ($requests.exceededMaxCharacters(message)) {
        // $rootScope.errors.push(
        //   'Comment exceeds 3000 characters'
        // );
        $window.alert('Comment exceeds 3000 characters');

        return false;
      }

      if (!message) {
        // $rootScope.errors.push(
        //   'Enter a reason why you\'re denying this request in ' +
        //   'the comment field.'
        // );

        $window.alert(
          'Enter a reason why you\'re denying this request in ' +
          'the comment field.'
        );

        return false;
      }

      return $api.update('requests/' + req.id + '/rejectdata', {
        data: {
          comments: message,
          createdBy: $user.id()
        }
      }).then(function (request) {

        if (request.success && request.response) {
          $window.alert(
            'This request\'s data has been rejected by the data manager.'
          );

          $scope.editable = false;
          $scope.delivered = false;
          $scope.submitted = false;
          $scope.sent = false;
          $scope.accepted = false;
          $scope.disableCopyTable();

          req.statusDescription = 'ResultsNotAccepted';

          $scope.loadLogs(request.response.id);
          $scope.resetForm();

          return true;
        } else {
          // $rootScope.errors.push(
          //   'Deny results request API call failed.');
          $api.errorAlert();
        }

        return false;
      });
    };

    $scope.submitComment = function (request) {
      //clear any existing errors
      $rootScope.errors = [];
      $rootScope.messages = [];

      if ($requests.exceededMaxCharacters(request.comments)) {
        // $rootScope.errors.push(
        //   'Comment exceeds 3000 characters'
        // );

        $window.alert('Comment exceeds 3000 characters');

        return false;
      }

      if (!request.comments) {
        // $rootScope.errors.push(
        //   'Please enter a comment and try again.'
        // );

        $window.alert(
          'Please enter a comment and try again.'
        );

        return false;
      }

      return $api.create('comment', {
        data: {
          requestId: request.id,
          status: request.statusDescription,
          comments: request.comments,
          createdBy: $user.id()
        }
      }).then(function (comment) {
        if (comment.success && comment.response) {
          // $rootScope.messages.push(
          //   'Comment posted.'
          // );
          $window.alert('Comment posted');

          $scope.loadLogs(comment.response.requestId);
          $scope.resetForm();

          return true;
        } else {
          // $rootScope.errors.push(
          //   'Failed to post comment, please try again.'
          // );
          $api.errorAlert();
        }

        return false;
      });
    };

    $scope.requestHistory = function (request) {
      //make sure they have an ID and the request exists
      if (request.id && parseInt(request.id, 10)) {
        $location.path('/requestHistory/' + request.id);
        return true;
      }

      return false;
    };

    $scope.setCommentability = function (request) {
      // Is user a Requestor or not?
      var currentStatus = request.statusDescription;
      if ($user.requester()) {
        if (
          currentStatus === 'Returned' ||
          currentStatus === 'Denied' ||
          currentStatus === 'RequestNotAccepted' ||
          currentStatus === 'ResultsAccepted' ||
          currentStatus === 'ResultsNotAccepted') {
          $scope.commentable = true;
        } else {
          $scope.commentable = false;
        }
      } else {
        if (
          currentStatus === 'Returned' ||
          currentStatus === 'Denied' ||
          currentStatus === 'RequestNotAccepted' ||
          currentStatus === 'ResultsAccepted' ||
          currentStatus === 'ResultsNotAccepted') {
          $scope.commentable = true;
        }
      }
    };

    $scope.setTableCopy = function (request) {
      var currentStatus = request.statusDescription;
      // This will change when we implement DSM / DDM requirements
      if ($user.destinationMgr() && currentStatus === 'ResultsDelivered') {
        $scope.canCopyTable = true;
      }
    };

    $scope.enableCopyTable = function () {
      $scope.copyTableEnabled = true;
      // Prefill data for copy table
      return $api.retrieve('copytable/domains', {})
        .then(function (copyTableAuth) {
          $scope.copyTableData = {};
          if (copyTableAuth.success && copyTableAuth.response) {
            $scope.copyTableData = copyTableAuth.response;
          }
        });
    };

    $scope.disableCopyTable = function () {
      $scope.canCopyTable = false;
    };

    $scope.collapseCopyTable = function () {
      $scope.clearCopyTableForm();
      $scope.copyTableEnabled = false;
    };

    $scope.clearCopyTableForm = function () {
      $scope.copyTableData.tableName = null;
      $scope.copyTableData.sourceSchemaName = null;
      $scope.copyTableData.destinationSchemaName = null;
      $scope.copyTableMessage = '';
    };

    $scope.submitCopyTableData = function (copyTableData) {
      $scope.canCopyTable = false;
      $scope.copyingTable = true;

      $rootScope.errors = [];
      $rootScope.messages = [];

      var baseErrorMessage = 'Please enter the following: ';
      var errorsMessage = '';

      if (!copyTableData.tableName) {
        // $rootScope.errors.push('Please enter a table name.');
        errorsMessage += 'Table name; ';
      }

      if (!copyTableData.sourceSchemaName) {
        errorsMessage += 'Source Schema Name; ';
      }

      if (!copyTableData.destinationSchemaName) {
        errorsMessage += 'Destination Schema Name;';
      }

      if (errorsMessage.length > 0) {
        $window.alert(baseErrorMessage + errorsMessage);
        $scope.canCopyTable = true;
        $scope.copyingTable = false;
        return false;
      }

      copyTableData.createdBy = $user.id();

      focus('copying');

      return $api.create('requests/' + $scope.request.id + '/copytable', {
        data: copyTableData
      }).then(function (copytable) {
        $scope.copyingTable = false;
        if (copytable.success && copytable.response) {
          $scope.loadLogs($scope.request.id);
          $scope.copyTableMessage = copytable.response;
          if (/Successfully/.test(copytable.response)) {
            $scope.successCopiedTable = true;
            $scope.copyTableMessage = copytable.response +
              '. Click here to proceed.';
            focus('success-copy');
          } else {
            $scope.failedCopiedTable = true;
            $scope.copyTableMessage = copytable.response +
              ' Click here to proceed.';
            focus('failed-copy');
          }
        } else {
          $scope.failedCopiedTable = true;
          $scope.copyTableMessage = 'An error has occured.' +
            ' Please contact system administrator.' +
            ' Click here to proceed.';
          focus('failed-copy');
        }
      });
    };

    $scope.copyTableProceed = function (clearable) {
      if (Boolean(clearable)) {
        $scope.clearCopyTableForm();
      }
      $scope.failedCopiedTable = false;
      $scope.successCopiedTable = false;
      $scope.canCopyTable = true;
    };

    $scope.canEdit = function (user, request) {
      if (user.id() === request.createdBy) {
        return true;
      } else {
        return false;
      }
    };

    // $scope.loadRequest($routeParams.id);
    $scope.loadRequestByID($routeParams.id);
    $scope.loadLogs($routeParams.id);

  }
]);